package com.agilex.healthcare.veteranappointment.restservice.appointments;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.veteranappointment.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequest;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestDetailCode;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestMessage;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestMessages;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequests;
import com.agilex.healthcare.veteranappointment.domain.VARDetailCode;
import com.agilex.healthcare.veteranappointment.domain.VARLinkTitles;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestType;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.veteranappointment.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.veteranappointment.testutility.TestHelper;

public class AppointmentRequestsResourceTest {

    private static MobileHealthClientTestVersion client;
    private static URI appointmentRequestsUri;
    private static VARAppointmentRequest mentalHealthAppointmentRequest1;
    private static VARAppointmentRequest secondAppointmentRequest;

    @BeforeClass
    public static void fixtureSetup() {
        client = TestHelper.createMobileHealthClient();

        appointmentRequestsUri = getAppointmentRequestsUri();

        cleanSubmittedRequests();
    }

    @AfterClass
    public static void tearDown(){

        if (client != null) {
            client.getJerseyClient().close();
            client = null;
        }
    }

    @Before
    public void appointmentRequestSetup(){
        mentalHealthAppointmentRequest1 = createValidAppointmentRequest();
        mentalHealthAppointmentRequest1 = client.getJerseyClient().target(appointmentRequestsUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(mentalHealthAppointmentRequest1, MediaType.APPLICATION_XML), VARAppointmentRequest.class);
    }

    private static void cleanSubmittedRequests() {
        //get all appointment requests for the patient
    	VARAppointmentRequests requests = retrieveAppointments(appointmentRequestsUri);

        for (VARAppointmentRequest request : requests) {
            if (request.getStatus().equals(AppointmentRequestStatus.SUBMITTED.getName())){
                cancelAppointmentRequest(request);
            }
        }
    }

    @After 
    public void appointmentRequestTearDown(){
        if (mentalHealthAppointmentRequest1 != null){
            mentalHealthAppointmentRequest1 =  cancelAppointmentRequest(mentalHealthAppointmentRequest1);
        }

        if (secondAppointmentRequest != null){
            secondAppointmentRequest = cancelAppointmentRequest(secondAppointmentRequest);
        }
    }

    private static VARAppointmentRequest cancelAppointmentRequest(VARAppointmentRequest appointmentRequest) {
        appointmentRequest.setActiveFlag(false);
        appointmentRequest.setStatus(AppointmentRequestStatus.CANCELLED.getName());
        String uriFormat = String.format("%s/appointment-service/patient/%s/%s/appointments/system/%s/id/%s",
				IntegrationTestConfiguration.getMobileHealthServerBaseUri(),"EDIPI","D123401","var",appointmentRequest.getAppointmentRequestId());
		URI uri = UriBuilder.fromUri(uriFormat).build();
        VARAppointmentRequest result = client.getJerseyClient().target(uri).request(MediaType.APPLICATION_XML).accept("application/xml").put(Entity.entity(appointmentRequest, MediaType.APPLICATION_XML), VARAppointmentRequest.class);
        return result;
    }

    @Test
    public void saveAppointmentRequest(){
        assertNotNull(mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId());
        assertNotNull(mentalHealthAppointmentRequest1.getSelfUri());

        assertEquals(mentalHealthAppointmentRequest1.isTextMessagingAllowed(), mentalHealthAppointmentRequest1.getPatient().isTextMessagingAllowed());
        assertEquals(mentalHealthAppointmentRequest1.getTextMessagingPhoneNumber(), mentalHealthAppointmentRequest1.getPatient().getTextMessagingPhoneNumber());
        assertEquals(true, mentalHealthAppointmentRequest1.isRequestedPhoneCall());
    }

    @Test
    public void retrieveAppointments() {
    	VARAppointmentRequests retrieveAppointments = retrieveAppointments(appointmentRequestsUri);
        assertTrue(retrieveAppointments.size()>0);
    }

    @Test
    public void retrieveAppointmentRequestsUsingDateFilter(){
        UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
        uriBuilder.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()));
        uriBuilder.queryParam("endDate", DateHelper.formatDate(DateHelper.getTomorrow()));
        URI uri = uriBuilder.build();
        VARAppointmentRequests retrieveAppointments = retrieveAppointments(uri);
        assertTrue(retrieveAppointments.size() > 0);
    }

    @Test
    public void retrieveEmptyAppointmentRequestsMessages(){
        UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
        uriBuilder.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()));
        uriBuilder.queryParam("endDate", DateHelper.formatDate(DateHelper.getTomorrow()));
        URI uri = uriBuilder.build();
        VARAppointmentRequests retrieveAppointments = retrieveAppointments(uri);
        URI relatedMessageUri = retrieveAppointments.get(0).getLink().getUriByTitle(VARLinkTitles.AppointmentRequestMessages);


        UriBuilder messageUriBuilder = UriBuilder.fromUri(relatedMessageUri);
        URI messageUri = messageUriBuilder.build();

        VARAppointmentRequestMessages messages = retrieveAppointmentRequestMessages(messageUri);
        assertEquals(0, messages.size());
        assertNotNull(messages);
    }

    @Test
    public void doNotSaveWithNonMatchingAppointmentRequestId(){

        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();
        String nonMatchingAppointmentRequestId = "0";
        int returnStatus = 0;
        int HTTP_STATUS_NULL_CONTENT=204;

        VARAppointmentRequestMessage message = new VARAppointmentRequestMessage();
        message.setAppointmentRequestId(nonMatchingAppointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }

        VARAppointmentRequestMessage savedAppointmentRequestMessage1 = null;
        //try{
            savedAppointmentRequestMessage1 = client.getJerseyClient().target(appointmentRequestMessagesUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(message, MediaType.APPLICATION_XML), VARAppointmentRequestMessage.class);
        /*} catch(UniformInterfaceException e){
            returnStatus = e.getResponse().getStatus();
        }*/

        //assertEquals(HTTP_STATUS_NULL_CONTENT, returnStatus);
        assertNull(savedAppointmentRequestMessage1);

    }

    @Test
    public void saveAppointmentRequestMessage(){
        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();

        VARAppointmentRequestMessage message = new VARAppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }
        VARAppointmentRequestMessage result = client.getJerseyClient().target(appointmentRequestMessagesUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(message, MediaType.APPLICATION_XML), VARAppointmentRequestMessage.class);

        assertNotNull(result);
        assertNotNull(result.getDataIdentifier());
        assertNotNull(result.getDataIdentifier().getUniqueId());
    }

    @Test
    public void retrieveAppointmentRequestMessagesTest(){

        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();

        VARAppointmentRequestMessage message = new VARAppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }
        VARAppointmentRequestMessage savedAppointmentRequestMessage1 = client.getJerseyClient().target(appointmentRequestMessagesUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(message, MediaType.APPLICATION_XML), VARAppointmentRequestMessage.class);
        assertNotNull(savedAppointmentRequestMessage1);

        VARAppointmentRequestMessages messages = retrieveAppointmentRequestMessages(appointmentRequestMessagesUri);

        assertNotNull(messages);
        assertEquals(1, messages.size());

        message = new VARAppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me again");
        message.setSenderId("D60000");
        message.setMessageDateTime(DateHelper.getToday());

        VARAppointmentRequestMessage savedAppointmentRequestMessage2 = client.getJerseyClient().target(appointmentRequestMessagesUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(message, MediaType.APPLICATION_XML), VARAppointmentRequestMessage.class);

        assertNotNull(savedAppointmentRequestMessage2);

        messages = retrieveAppointmentRequestMessages(appointmentRequestMessagesUri);

        assertNotNull(messages);
        assertEquals(messages.size(), 2);

    }


    @Test
    public void updateAppointmentRequest(){
        mentalHealthAppointmentRequest1.setStatus(AppointmentRequestStatus.BOOKED.getName());
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE1");
        Date beforeUpdate = DateHelper.getYesterday();
        mentalHealthAppointmentRequest1.setLastUpdatedDate(beforeUpdate);
        mentalHealthAppointmentRequest1 = client.getJerseyClient().target(mentalHealthAppointmentRequest1.getSelfUri()).request(MediaType.APPLICATION_XML).accept("application/xml").put(Entity.entity(mentalHealthAppointmentRequest1, MediaType.APPLICATION_XML), VARAppointmentRequest.class);
        assertNotNull(mentalHealthAppointmentRequest1);
        assertNotNull(AppointmentRequestStatus.BOOKED.getName(), mentalHealthAppointmentRequest1.getStatus());
        assertTrue(mentalHealthAppointmentRequest1.getLastUpdatedDate().after(beforeUpdate));
    }

    @Test
    public void updateAppointmentRequestWithMultipleDetailCodes(){
        mentalHealthAppointmentRequest1.setStatus(AppointmentRequestStatus.CANCELLED.getName());
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE8");
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE7");
        Date beforeUpdate = DateHelper.getYesterday();
        mentalHealthAppointmentRequest1.setLastUpdatedDate(beforeUpdate);
        mentalHealthAppointmentRequest1 = client.getJerseyClient().target(mentalHealthAppointmentRequest1.getSelfUri()).request(MediaType.APPLICATION_XML).accept("application/xml").put(Entity.entity(mentalHealthAppointmentRequest1, MediaType.APPLICATION_XML), VARAppointmentRequest.class);
        assertNotNull(mentalHealthAppointmentRequest1);
        assertNotNull(AppointmentRequestStatus.CANCELLED.getName(), mentalHealthAppointmentRequest1.getStatus());
        assertTrue(mentalHealthAppointmentRequest1.getLastUpdatedDate().after(beforeUpdate));
    }

    @Test
    public void validateSaveSecondAppointmentRequest(){
        secondAppointmentRequest = createValidAppointmentRequest();
        secondAppointmentRequest.setParentRequest(mentalHealthAppointmentRequest1);
        secondAppointmentRequest.setSecondRequest(true);
        mentalHealthAppointmentRequest1.setSecondRequestSubmitted(true);

        secondAppointmentRequest = 	client.getJerseyClient().target(appointmentRequestsUri).request(MediaType.APPLICATION_XML).accept("application/xml").post(Entity.entity(secondAppointmentRequest, MediaType.APPLICATION_XML), VARAppointmentRequest.class);

        assertEquals(true, secondAppointmentRequest.isSecondRequest());
        assertNull(secondAppointmentRequest.getParentRequest());

        VARAppointmentRequest parentRequest = client.getJerseyClient().target(mentalHealthAppointmentRequest1.getSelfUri()).request(MediaType.APPLICATION_XML).accept("application/xml").get(VARAppointmentRequest.class);
        assertEquals(true, parentRequest.isSecondRequestSubmitted());
    }

    @Test
    public void validateSetProviderSeenAppointmentRequestResourceIsAvailable(){
    	//Not possible to test Provider specific resource in va-veteran-dev mode. Verifying if the Resource is available or not.
    	//There is seperate unit test at DataLayer level.
    	assertNotNull(mentalHealthAppointmentRequest1.getLink().getUriByTitle(VARLinkTitles.AppointmentRequestProviderSeenFlag));
    }

	private void addARDetailCode(VARAppointmentRequest appointmentRequest, String code){

		VARDetailCode detailCode = new VARDetailCode();
        detailCode.setCode(code);

        List<VARAppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
        VARAppointmentRequestDetailCode arDetailCode = new VARAppointmentRequestDetailCode();
        arDetailCode.setCreatedDate(new Date());
        arDetailCode.setDetailCode(detailCode);
        arDetailCode.setUserId("zztest.staff01");
        arDetailCodes.add(arDetailCode);
        appointmentRequest.setAppointmentRequestDetailCode(arDetailCodes);
    }

    private static VARAppointmentRequests retrieveAppointments(URI appointmentsBaseUri) {
        VARAppointmentRequests appointmentRequests =
                client.getJerseyClient().target(appointmentsBaseUri).request(MediaType.APPLICATION_JSON)
                        .accept(MediaType.APPLICATION_JSON).get(VARAppointmentRequests.class);

        return appointmentRequests;
    }

    private static URI getAppointmentRequestsUri() {
        /*ResourceDirectory resourceDirectory = ResourceDirectoryLoader.loadResourceDirectory();
        URI mhpUserUri = resourceDirectory.getLink().getUriByTitle(LinkTitles.MHPUser);

        MhpUser mhpUser = client.getJerseyClient().resource(mhpUserUri).get(MhpUser.class);

        URI appointmentRequestsUri = mhpUser.getLink().getUriByTitle(LinkTitles.Appointments);

        return appointmentRequestsUri;*/
    	String uriFormat = String.format("%s/appointment-service/patient/%s/%s/appointments",
				IntegrationTestConfiguration.getMobileHealthServerBaseUri(),"EDIPI","D123401");
		URI uri = UriBuilder.fromUri(uriFormat).build();
		return uri;
    }

    private static URI getAppointmentRequestMessagesUri(String appointmentRequestId) {
        URI appointmentRequestsUri = getAppointmentRequestsUri();
        VARAppointmentRequests appointmentRequests = client.getJerseyClient().target(appointmentRequestsUri).request().get(VARAppointmentRequests.class);

        for (VARAppointmentRequest request : appointmentRequests) {
            if(request.getAppointmentRequestId().equals(appointmentRequestId)){
                return request.getLink().getUriByTitle(VARLinkTitles.AppointmentRequestMessages);
            }
        }
        return null;
    }

    private VARAppointmentRequestMessages retrieveAppointmentRequestMessages(URI uri) {
        VARAppointmentRequestMessages messages = client.getJerseyClient().target(uri).request(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_JSON).get(VARAppointmentRequestMessages.class);
        return messages;
    }

    private static VARAppointmentRequest createValidAppointmentRequest() {
        String appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
        String email = "test@agilex.com";
        String phoneNumber = "+1 (555) 555-5555 x55.55";
        String visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
        String purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
        String otherPurposeOfVisit = "Other purpose of visit specified test";
        String providerId = "PROV1";
        boolean secondRequest = false;

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String optionDate1 = DateHelper.formatDate(cal.getTime());
        String optionTime1 = "AM";
        cal.add(Calendar.DATE, 1);
        String optionDate2 = DateHelper.formatDate(cal.getTime());
        String optionTime2 = "PM";
        cal.add(Calendar.DATE, 2);
        String optionDate3 = DateHelper.formatDate(cal.getTime());
        String optionTime3 = "AM";
        HashSet<String> bestTimeToCall = new HashSet<String>();
        bestTimeToCall.add("9 AM - 11 AM");
        bestTimeToCall.add("11 AM - 1 PM");
        String textMessagingPhoneNumber = "555-222-3242";
        boolean textMessagingAllowed = true;
        boolean requestedPhoneCall = true;
        String facilityAddress = "1201 Broad Block Blvd Richmond, VA 23249";
        String facilityCity = "Richmond";
        String facilityState = "VA";
        String facilityCode = "652";
        String facilityName = "Richmond VAMC";
        String facilityParentSiteCode = "688";
        String facilityType = "VAMC";
        Facility facility = createFacility(facilityAddress, facilityCity, facilityCode, facilityName, facilityParentSiteCode, facilityState, facilityType);

        VARAppointmentRequest a = new VARAppointmentRequest();

        a.setAppointmentType(appointmentType);
        a.setVisitType(visitType);
        a.setFacility(facility);
        a.setEmail(email);
        a.setPhoneNumber(phoneNumber);
        a.setOptionDate1(optionDate1);
        a.setOptionTime1(optionTime1);
        a.setOptionDate2(optionDate2);
        a.setOptionTime2(optionTime2);
        a.setOptionDate3(optionDate3);
        a.setOptionTime3(optionTime3);
        a.setBestTimetoCall(bestTimeToCall);
        a.setPurposeOfVisit(purposeOfVisit);
        a.setOtherPurposeOfVisit(otherPurposeOfVisit);
        a.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
        a.setProviderId(providerId);
        a.setLastUpdatedDate(new Date());
        a.setSecondRequest(secondRequest);
        a.setProviderName("zztest staff01");
        a.setTextMessagingAllowed(textMessagingAllowed);
        a.setTextMessagingPhoneNumber(textMessagingPhoneNumber);
        a.setRequestedPhoneCall(requestedPhoneCall);
        a.setTypeOfCareId("502");
        return a;
    }

    private static Facility createFacility(String facilityAddress, String facilityCity, String facilityCode, String facilityName, String facilityParentSiteCode, String facilityState, String facilityType) {
        Facility facility = new Facility();
        facility.setAddress(facilityAddress);
        facility.setCity(facilityCity);
        facility.setFacilityCode(facilityCode);
        facility.setName(facilityName);
        facility.setParentSiteCode(facilityParentSiteCode);
        facility.setState(facilityState);
        facility.setType(facilityType);

        return facility;
    }


}
